/*===========================================================================*/
/* SORENSEN SGI                                                              */
/* LabWindows/CVI Instrument Driver                                          */
/* Original Release:                                                         */
/*                                                                           */
/* Modification History: None                                                */
/*===========================================================================*/

#include <visa.h>
#include <userint.h>
#include <ansi_c.h>
#include <formatio.h>
#include <string.h>

#include "SNSGI.h"

#define SNSGI_REVISION   "V1.01, 6/2004, CVI 5.5.1"    /* Driver Revision */
#define BUFFER_SIZE      256L                          /* File I/O buffer */

#define DEMO_IF			0
#define GPIB_IF			1
#define RS232_IF		2
#define ENET_IF			3

#define LOCK                                                      \
    if ((SNSGI_status = viLock(instrSession,					  \
                                  VI_EXCLUSIVE_LOCK,			  \
                                  THREAD_LOCK_TIMEOUT,			  \
                                  NULL,NULL)) < 0)                \
        return SNSGI_status;                                      \

#define UNLOCK                                                    \
{                                                                 \
    if((unlockStatus = viUnlock(instrSession)) < 0)               \
        return unlockStatus;                                      \
    return SNSGI_status;                                          \
}

/*****************************************************************************/
/*= INSTRUMENT-DEPENDENT STATUS/RANGE STRUCTURE  ============================*/
/*****************************************************************************/
/* SNSGI_stringValPair is used in the SNSGI_errorMessage function          */
/* SNSGI_statusDataRanges is used to track session dependent status & ranges*/
/*===========================================================================*/
typedef struct  SNSGI_stringValPair
{
   ViStatus stringVal;
   ViString stringName;
}  SNSGI_tStringValPair;
 

typedef struct SNSGI_statusDataRanges *SNSGI_instrRange;

static short int		user_if;  

/*****************************************************************************/
/*= UTILITY ROUTINE DECLARATIONS (Non-Exportable Functions) =================*/
/*****************************************************************************/
ViBoolean SNSGI_invalidViBooleanRange (ViBoolean val);
ViBoolean SNSGI_invalidViInt16Range   (ViInt16 val, ViInt16 min, ViInt16 max);
ViBoolean SNSGI_invalidViInt32Range   (ViInt32 val, ViInt32 min, ViInt32 max);
ViBoolean SNSGI_invalidViUInt8Range   (ViUInt8 val, ViUInt8 min, ViUInt8 max);
ViBoolean SNSGI_invalidViUInt16Range  (ViUInt16 val, ViUInt16 min, ViUInt16 max);
ViBoolean SNSGI_invalidViUInt32Range  (ViUInt32 val, ViUInt32 min, ViUInt32 max);
ViBoolean SNSGI_invalidViReal32Range  (ViReal32 val, ViReal32 min, ViReal32 max);
ViBoolean SNSGI_invalidViReal64Range  (ViReal64 val, ViReal64 min, ViReal64 max);
ViStatus  SNSGI_initCleanUp           (ViSession openRMSession,
                                          ViPSession openInstrSession,
                                          ViStatus currentStatus);

/*****************************************************************************/
/*====== USER-CALLABLE FUNCTIONS (Exportable Functions) =====================*/
/*****************************************************************************/

/*===========================================================================*/
/* Function: Initialize                                                      */
/* Purpose:  This function opens the instrument, queries the instrument      */
/*           for its ID, and initializes the instrument to a known state.    */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_init (ViRsrc     resourceName,
                                ViBoolean  IDQuery,
                                ViBoolean  resetDevice,
                                ViPSession instrSession)
{
    ViStatus	SNSGI_status,unlockStatus;
    ViSession	rmSession = 0;
    ViUInt32	retCnt = 0;
    ViInt16		i;
    ViByte		rdBuffer[BUFFER_SIZE];
    ViByte		tmpBuffer[10];

    /*- Open instrument session ---------------------------------------------*/
    user_if = GPIB_IF;
    if ((SNSGI_status = viOpenDefaultRM (&rmSession)) < 0)
    {
    	sprintf(rdBuffer,"viOpenDefaultRM - %08X",SNSGI_status);
		MessagePopup("SNSGI Error",rdBuffer);
        return SNSGI_status;
    }

    if ((SNSGI_status = viOpen (rmSession, resourceName, VI_NULL, VI_NULL, instrSession)) < 0)
    {
    	sprintf(rdBuffer,"viOpen - %08X",SNSGI_status);
		MessagePopup("SNSGI Error",rdBuffer);
        viClose (rmSession);
        return SNSGI_status;
    }
	viSetAttribute (*instrSession, VI_ATTR_TMO_VALUE, 3000);

	/* set up serial port if selected */
	if (strstr (resourceName, "ASRL") != NULL)
	{
		user_if = RS232_IF;
		viSetAttribute (*instrSession, VI_ATTR_ASRL_BAUD, 19200);
		viSetAttribute (*instrSession, VI_ATTR_ASRL_DATA_BITS, 8);
		viSetAttribute (*instrSession, VI_ATTR_ASRL_PARITY, VI_ASRL_PAR_NONE);
		viSetAttribute (*instrSession, VI_ATTR_ASRL_STOP_BITS, VI_ASRL_STOP_ONE);
		viSetAttribute (*instrSession, VI_ATTR_ASRL_FLOW_CNTRL, VI_ASRL_FLOW_NONE);
		viSetAttribute (*instrSession, VI_ATTR_ASRL_END_OUT, VI_ASRL_END_TERMCHAR);
		viSetAttribute (*instrSession, VI_ATTR_TERMCHAR, '\r');
	}
	else if ((resourceName, "TCPIP") != NULL)
	{
		user_if = ENET_IF;
		/*
		viSetAttribute (*instrSession, VI_ATTR_TERMCHAR_EN, VI_TRUE);
		viSetAttribute (*instrSession, VI_ATTR_TERMCHAR, '\n');
		*/
	}
	
    /*- Identification Query ------------------------------------------------*/
    if (IDQuery)
    {
		if ((SNSGI_status = SNSGI_idQuery (*instrSession,rdBuffer)) < VI_SUCCESS)
            return SNSGI_initCleanUp (rmSession, instrSession, SNSGI_status);

		/* get and compare first eight bytes from *idn? response */
		for (i = 0; i < 8; ++i)
			tmpBuffer[i] = rdBuffer[i];
		tmpBuffer[8] = '\0';
		StringUpperCase (tmpBuffer);
		if (strcmp(tmpBuffer,"SORENSEN") != 0)
            return SNSGI_initCleanUp (rmSession, instrSession, VI_ERROR_FAIL_ID_QUERY);
    }
        
    /*- Reset instrument ----------------------------------------------------*/
    if (resetDevice)
    {
        if ((SNSGI_status = SNSGI_reset (*instrSession)) != VI_SUCCESS)
            return SNSGI_initCleanUp (rmSession, instrSession, SNSGI_status);
    }       
          
    return SNSGI_status;
}

/*===========================================================================*/
/* Function: GenerateDCVolt                                                  */
/* Purpose:  This function sets the output voltage to the specified value.   */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_GenerateDCVolt (ViSession      instrSession,
                                       ViReal64       voltage)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK

    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE:VOLTAGE ");
    sprintf(tmpBuffer,"%f",voltage);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
}

/*===========================================================================*/
/* Function: GenerateDCCurr                                                  */
/* Purpose:  This function sets the output current to the specified value.   */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_GenerateDCCurr (ViSession      instrSession,
                                       ViReal64       current)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE:CURRENT ");
    sprintf(tmpBuffer,"%f",current);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
}
/*===========================================================================*/
/* Function: SNSGI_GeneratePower                                                  */
/* Purpose:  This function sets the output power to the specified value.   */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_GeneratePower (ViSession instrSession, ViReal64 watts)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE:POWER ");
    sprintf(tmpBuffer,"%f",watts);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
}

/*===========================================================================*/
/* Function: GenerateDCVRamp                                                 */
/* Purpose:  This function initates a output voltage ramp from the present   */
/*           output voltage to the specified output voltage in the specified */
/*           time on the specified channel. Time is in seconds.              */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_GenerateDCVRamp  (ViSession      instrSession,
                                         ViReal64       voltage,
                                         ViReal64       time_sec)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE:VOLTAGE:RAMP ");
    sprintf(tmpBuffer,"%f ",voltage);
    strcat(cmdBuffer,tmpBuffer);
    sprintf(tmpBuffer,"%f",time_sec);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
}
                                          
/*===========================================================================*/
/* Function: GenerateDCCRamp                                                 */
/* Purpose:  This function initates a output Current ramp from the present   */
/*           output voltage to the specified output current in the specified */
/*           time on the specified channel. Time is in seconds.              */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_GenerateDCCRamp  (ViSession      instrSession,
                                         ViReal64       current,
                                         ViReal64       time_sec)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE:CURRENT:RAMP ");
    sprintf(tmpBuffer,"%f ",current);
    strcat(cmdBuffer,tmpBuffer);
    sprintf(tmpBuffer,"%f",time_sec);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
}

/*===========================================================================*/
/* Function: OutState                                                       */
/* Purpose:  This function reads the output voltage.                         */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_OutState (ViSession      instrSession,
                                 ViBoolean      state)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"OUTPUT:STATE ");
    sprintf(tmpBuffer,"%d",state);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
    
}

/*===========================================================================*/
/* Function: MeasureDCVolt                                                   */
/* Purpose:  This function reads the output voltage.                         */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_MeasureDCVolt (ViSession      instrSession,
                                      ViReal64       *voltage)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"MEASURE:VOLTAGE?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query */
	if ((SNSGI_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *voltage = atof(rdBuffer);

	UNLOCK
}

/*===========================================================================*/
/* Function: MeasureDCCurr                                                   */
/* Purpose:  This function reads the output voltage.                         */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_MeasureDCCurr (ViSession      instrSession,
                                      ViReal64       *current)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"MEASURE:CURRENT?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* get returned query */
    if ((SNSGI_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *current = atof(rdBuffer);

    UNLOCK
}
/*===========================================================================*/
/* Function: SNSGI_MeasurePower                                              */
/* Purpose:  This function reads the output power.                           */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_MeasurePower (ViSession instrSession,
                             ViReal64 *powerMeasurement)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"MEASURE:POWER?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* get returned query */
    if ((SNSGI_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *powerMeasurement = atof(rdBuffer);

    UNLOCK   
}

/*===========================================================================*/
/* Function: Mode                                                            */
/* Purpose:  This function returns the operating mode of the power supply.   */
/*           (voltage mode = 0, current mode = 1)                            */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ModeQuery (ViSession      instrSession,
                                  ViInt16        *mode)
{
    ViStatus		SNSGI_status,unlockStatus;
	unsigned char	protCondReg  = 0;
	ViByte			statusMessage[BUFFER_SIZE];

	LOCK
	
	if ((SNSGI_status = SNSGI_ProtCondQuery(instrSession, &protCondReg, statusMessage)) < 0)
	{
		*mode = -1;
		UNLOCK;
	}

	if (protCondReg & SNSGI_PE_CURRMODE)
		*mode = 1;
	else
		*mode = 0;
		
	UNLOCK
}

/*===========================================================================*/
/* Function: VoltMode                                                        */
/* Purpose:  This function configures the power supply to shutdown if the    */
/*           power supply switches from voltage mode to current mode.        */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_VoltMode (ViSession      instrSession,
                                 ViBoolean      serviceRequest)
                                
{
    ViStatus		SNSGI_status,unlockStatus;
	unsigned char	mask;

	LOCK
	
	mask =  SNSGI_PE_CURRMODE | SNSGI_PE_FOLD;
	
	/* program foldback */
	if ((SNSGI_status = SNSGI_ConfigFoldback (instrSession, SNSGI_FB_CURRMODE)) < 0)
		UNLOCK
		
	/* program protection event enable mask for foldback and current mode */
	if ((SNSGI_status = SNSGI_ConfigProtEventEnable (instrSession, mask)) < 0)
		UNLOCK
		
	if (serviceRequest)
	{
		/* program service request enable mask */
		SNSGI_status = SNSGI_ConfigSerReqEnable (instrSession, SNSGI_SB_SRQ);
	}
	
	UNLOCK
}

/*===========================================================================*/
/* Function: CurrMode                                                        */
/* Purpose:  This function configures the power supply to shutdown if the    */
/*           power supply switches from current mode to voltage mode.        */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_CurrMode (ViSession      instrSession,
                                 ViBoolean      serviceRequest)
                                
{
    ViStatus	SNSGI_status,unlockStatus;
	unsigned char	mask;

	LOCK
	
	mask =  SNSGI_PE_VOLTMODE | SNSGI_PE_FOLD;
	
	/* program foldback */
	if ((SNSGI_status = SNSGI_ConfigFoldback (instrSession, SNSGI_FB_VOLTMODE)) < 0)
		UNLOCK
		
	/* program protection event enable mask */
	if ((SNSGI_status = SNSGI_ConfigProtEventEnable (instrSession, mask)) < 0)
		UNLOCK
		
	if (serviceRequest)
	{
		/* program service request enable mask */
		SNSGI_status = SNSGI_ConfigSerReqEnable (instrSession, SNSGI_SB_SRQ);
	}
	
	UNLOCK    
}

/*===========================================================================*/
/* Function: NormalMode                                                      */
/* Purpose:  This function configures the power supply operate in both       */
/*           current mode and voltage mode. All protection events and        */
/*           service request enables are cleared.                            */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_NormalMode (ViSession      instrSession)
{
    ViStatus	SNSGI_status,unlockStatus;

	LOCK
	
	/* program foldback */
	if ((SNSGI_status = SNSGI_ConfigFoldback (instrSession, SNSGI_FB_NORMAL)) < 0)
		UNLOCK
		
	/* program protection event enable mask */
	if ((SNSGI_status = SNSGI_ConfigProtEventEnable (instrSession, 0)) < 0)
		UNLOCK
		
	/* program service request enable mask */
	SNSGI_status = SNSGI_ConfigSerReqEnable (instrSession, 0);

	UNLOCK    
}

/*===========================================================================*/
/* Function: ConfigDCVolt                                                    */
/* Purpose:  This function configures the power supply for a triggered       */
/*           output voltage of the specified value. A software trigggger     */
/*           must be sent to the power supply to implement this sequence.    */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ConfigDCVolt (ViSession      instrSession,
                                     ViReal64       voltage)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];
    
	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE:VOLTAGE:TRIGGERED ");
    sprintf(tmpBuffer,"%f",voltage);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}
                                          
/*===========================================================================*/
/* Function: ConfigDCCurr                                                    */
/* Purpose:  This function configures the power supply for a triggered       */
/*           output voltage of the specified value. A software trigggger     */
/*           must be sent to the power supply to implement this sequence.    */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ConfigDCCurr (ViSession      instrSession,
                                     ViReal64       current)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];
    
	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE:CURRENT:TRIGGERED ");
    sprintf(tmpBuffer,"%f",current);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigDCVRamp                                                   */
/* Purpose:  This function configures the power supply for a triggered       */
/*           output voltage ramp from the present output voltage to the      */
/*           specified voltage in the specified time on the specified        */
/*           channel. A software trigger must be sent to the power supply to */
/*           implement this sequence.                                        */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ConfigDCVRamp (ViSession      instrSession,
                                      ViReal64       voltage,
                                      ViReal64       time_sec)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];
    
	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE:VOLTAGE:RAMP:TRIGGERED ");
    sprintf(tmpBuffer,"%f ",voltage);
    strcat(cmdBuffer,tmpBuffer);
    sprintf(tmpBuffer,"%f",time_sec);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}
                                          
/*===========================================================================*/
/* Function: ConfigDCCRamp                                                   */
/* Purpose:  This function configures the power supply for a triggered       */
/*           output current ramp from the present output current to the      */
/*           specified current in the specified time on the specified        */
/*           channel. A software trigger must be sent to the power supply to */
/*           implement this sequence. Time is in seconds.                    */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ConfigDCCRamp (ViSession      instrSession,
                                      ViReal64       current,
                                      ViReal64       time_sec)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];
    
	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE:CURRENT:RAMP:TRIGGERED ");
    sprintf(tmpBuffer,"%f ",current);
    strcat(cmdBuffer,tmpBuffer);
    sprintf(tmpBuffer,"%f",time_sec);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigOVP                                                       */
/* Purpose:  This function sets the output overvoltage protection to the     */
/*           specified value.                                                */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ConfigOVP (ViSession      instrSession,
                                  ViReal64       voltage)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];
    
	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE:VOLTAGE:PROTECTION ");
    sprintf(tmpBuffer,"%f",voltage);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);

    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigSoftLimits                                                */
/* Purpose:  This function sets the soft voltage and current programming     */
/*           limits to the specified values.                                 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ConfigSoftLimits (ViSession      instrSession,
                                         ViReal64       voltage,
                                         ViReal64       current)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];
    
	LOCK
	
    /* build and send SCPI command string - voltage */
    strcpy(cmdBuffer,"SOURCE:VOLTAGE:LIMIT ");
    sprintf(tmpBuffer,"%f",voltage);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* build and send SCPI command string - current */
    strcpy(cmdBuffer,"SOURCE:CURRENT:LIMIT ");
    sprintf(tmpBuffer,"%f",current);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigPowerON                                                   */
/* Purpose:  This function configures the output power on values for the     */
/*           specified channel.                                              */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ConfigPowerOn (ViSession		instrSession,
                                      ViReal64		voltage,
                                      ViReal64		current,
                                      ViReal64		ovp)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK
	
    /* build and send SCPI command string - voltage */
    strcpy(cmdBuffer,"CALIBRATE:INITIAL:VOLTAGE ");
    sprintf(tmpBuffer,"%f",voltage);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);

    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* build and send SCPI command string - current */
    strcpy(cmdBuffer,"CALIBRATE:INITIAL:CURRENT ");
    sprintf(tmpBuffer,"%f",current);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);

    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* build and send SCPI command string - ovp */
    strcpy(cmdBuffer,"CALIBRATE:INITIAL:VOLTAGE:PROTECTION ");
    sprintf(tmpBuffer,"%f",ovp);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);

    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* build and send SCPI command string - unlock */
    strcpy(cmdBuffer,"CALIBRATE:UNLOCK \"6867\"");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* build  and send SCPI command string - store */
    strcpy(cmdBuffer,"CALIBRATE:STORE");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* build and send SCPI command string - lock */
    strcpy(cmdBuffer,"CALIBRATE:LOCK");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigFoldback                                                  */
/* Purpose:  This function configures the foldback mode if any. Three modes  */
/*           are normal, constant current, and constant voltage.             */
/*           See header file for bit definitions.                            */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ConfigFoldback (ViSession      instrSession,
                                       ViInt16        mode)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"OUTPUT:PROTECTION:FOLD ");
    sprintf(tmpBuffer,"%d",mode);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigProtDelay                                                 */
/* Purpose:  This function configures the programmable delay which is        */
/*           executed prior to reporting any output protection conditions.   */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ConfigProtDelay (ViSession instrSession,
                                        ViReal64  delay)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"OUTPUT:PROTECTION:DELAY ");
    sprintf(tmpBuffer,"%f",delay);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigVoltAve                                                   */
/* Purpose:  This function configures the number of current measurents to be */
/*           averaged. 1-5                                                   */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ConfigVoltAve (ViSession instrSession,
                                      ViInt16   average)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"MEASURE:VOLTAGE:AVERAGE ");
    sprintf(tmpBuffer,"%d",average);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigCurrAve                                                   */
/* Purpose:  This function configures the number of current measurents to be */
/*           averaged. 1-5                                                   */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ConfigCurrAve (ViSession instrSession,
                                      ViInt16   average)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"MEASURE:CURRENT:AVERAGE ");
    sprintf(tmpBuffer,"%d",average);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigProtEventEnable                                           */
/* Purpose:  This function configures the protection event status enable     */
/*           register. See header file for bit definitions.                  */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ConfigProtEventEnable (ViSession      instrSession,
                                              unsigned char  protectionEventEnable)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"STATUS:PROTECTION:ENABLE ");
    sprintf(tmpBuffer,"%d",protectionEventEnable);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigStandEventEnable                                          */
/* Purpose:  This function configures the standard event status enable       */
/*           register. See header file for bit definitions.                  */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ConfigStandEventEnable (ViSession      instrSession,
                                               unsigned char  standardEventEnable)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"*ESE");
    sprintf(tmpBuffer," %d",standardEventEnable);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigSerReqEnable                                              */
/* Purpose:  This function configures the service request enable             */
/*           register. See header file for bit definitions.                  */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ConfigSerReqEnable (ViSession      instrSession,
                                           unsigned char  serviceRequestEnable)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"*SRE");
    sprintf(tmpBuffer," %d",serviceRequestEnable);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: OutputVoltQuery                                                */
/* Purpose:  This function returns the programmed output voltage value.      */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_OutputVoltQuery  (ViSession      instrSession,
                                         ViReal64       *voltage)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE:VOLTAGE?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* get returned query string */
    if ((SNSGI_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *voltage = atof(rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: OutputCurrQuery                                                */
/* Purpose:  This function returns the programmed output voltage value.      */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_OutputCurrQuery  (ViSession      instrSession,
                                         ViReal64       *current)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE:CURRENT?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

	/* get returnd query string */
    if ((SNSGI_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *current = atof(rdBuffer);

    UNLOCK
}
/*===========================================================================*/
/* Function: OutputPowerQuery                                                */
/* Purpose:  This function returns the programmed output power value.        */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_OutputPowerQuery (ViSession instrSession,
                                 ViReal64 *ProgrammedPower)
{
   ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE:POWER?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

	/* get returnd query string */
    if ((SNSGI_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *ProgrammedPower = atof(rdBuffer);

    UNLOCK   
}

/*===========================================================================*/
/* Function: OutputOVPQuery                                                  */
/* Purpose:  This function returns the programmed output voltage value.      */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_OutputOVPQuery (ViSession      instrSession,
                                       ViReal64       *OVP)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE:VOLTAGE:PROTECTION?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get return query string */
    if ((SNSGI_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *OVP = atof(rdBuffer);

    UNLOCK
}
                                        

/*===========================================================================*/
/* Function: OutStateQuery                                                   */
/* Purpose:  This function returns the output state, 1 = ON.                 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_OutStateQuery (ViSession      instrSession,
                                      ViBoolean      *state)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string - voltage limit */
    strcpy(cmdBuffer,"OUTPUT:STATE?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* get returned query string */
    if ((SNSGI_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *state = atoi(rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: SoftLimitsQuery                                                 */
/* Purpose:  This function returns the voltage and current soft limits.      */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_SoftLimitsQuery  (ViSession      instrSession,
                                         ViReal64       *voltage,
                                         ViReal64       *current)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string - voltage limit */
    strcpy(cmdBuffer,"SOURCE:VOLTAGE:LIMIT?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* get returned query string */
    if ((SNSGI_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *voltage = atof(rdBuffer);

	/* build and send SCPI command string - current limit */
    strcpy(cmdBuffer,"SOURCE:CURRENT:LIMIT?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query string */
    if ((SNSGI_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *current = atof(rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: PowerONQuery                                                    */
/* Purpose:  This function returns the output power on values for the        */
/*           specified channel.                                              */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_PowerOnQuery  (ViSession      instrSession,
                                      ViReal64       *voltage,
                                      ViReal64       *current,
                                      ViReal64       *ovp)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];
    
	LOCK
	
	/* build and send SCPI command string - power on voltage */
    strcpy(cmdBuffer,"CALIBRATE:INITIAL:VOLTAGE?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query string */
    if ((SNSGI_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *voltage = atof(rdBuffer);

	/* build and send SCPI comand string - power on current */
    strcpy(cmdBuffer,"CALIBRATE:INITIAL:CURRENT?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query string */
    if ((SNSGI_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *current = atof(rdBuffer);

	/* build and send SCPI command string - power on ovp */
    strcpy(cmdBuffer,"CALIBRATE:INITIAL:VOLTAGE:PROTECTION?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query string */
    if ((SNSGI_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *ovp = atof(rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: protDelayQuery                                                  */
/* Purpose:  This function returns the protection condition delay value      */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_protDelayQuery (ViSession instrSession,
                                       ViReal64  *delay)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string - voltage limit */
    strcpy(cmdBuffer,"OUTPUT:PROTECTION:DELAY?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* get returned query string */
    if ((SNSGI_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *delay = atof(rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: measVoltAveQuery                                                */
/* Purpose:  This function returns the number of voltage readings that will  */
/*           be averaged upon receipt of a voltage measurent command for the */
/*           specified channel.                                              */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_measVoltAveQuery (ViSession instrSession,
                                         ViInt16   *average)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string - voltage limit */
    strcpy(cmdBuffer,"MEASURE:VOLTAGE:AVERAGE?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* get returned query string */
    if ((SNSGI_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *average = atoi(rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: measCurrAveQuery                                                */
/* Purpose:  This function returns the number of current readings that will  */
/*           be averaged upon receipt of a current measurent command for the */
/*           specified channel.                                              */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_measCurrAveQuery (ViSession instrSession,
                                         ViInt16   *average)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string - voltage limit */
    strcpy(cmdBuffer,"MEASURE:CURRENT:AVERAGE?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* get returned query string */
    if ((SNSGI_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *average = atoi(rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: SeqInit                                                		 */
/* Purpose:  This routine checks to see if system has finished its power-on	 */
/*	 		 initalization to be able to work with sequences.  The query	 */
/*			 returns 1 until the system finishes at which point 0 is		 */
/*			 returned.														 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_SeqInit (ViSession instrSession, ViInt16 *initialized)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string - voltage limit */
    strcpy(cmdBuffer,"PROGRAM:INITIALIZING?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* get returned query string */
    if ((SNSGI_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *initialized =  atoi(rdBuffer);

    UNLOCK
    
}

/*===========================================================================*/
/* Function: SeqCatalog                                                		 */
/* Purpose:  This routine returns the names of all the defined sequences.    */
/*																			 */
/*																			 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_SeqCatalog (ViSession instrSession, ViChar catalog[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViChar		rdBuffer[BUFFER_SIZE];
    
	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"PROGRAM:CATALOG?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);

    if (SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &sendCnt) < 0)
    	UNLOCK

    /* get returned query string */
    if ((SNSGI_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
	
	strcpy(catalog, rdBuffer);

    UNLOCK

}

/*===========================================================================*/
/* Function: SeqArm	                                                		 */
/* Purpose:  This routine sets up the selected sequence for triggered		 */
/*			 operation. See the Trigger Sequence command.					 */
/*			 																 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_SeqArm (ViSession instrSession)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"PROGRAM:ARM");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
}

/*===========================================================================*/
/* Function: ProgVIMode                                                		 */
/* Purpose:  This routine programs the selected step for Voltage or Current	 */
/*			 Mode operation at the defined Voltage, Current, and OVP values	 */
/*			 for the specified time. The mode of the power supply is settings*/
/*			 and load dependent.											 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ProgVIMode (ViSession instrSession, ViInt16 step,
                           ViReal64 voltage, ViReal64 current, ViReal64 OVP,
                           ViReal64 time)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"PROGRAM:DEFINE ");
    sprintf(tmpBuffer,"%d VIMODE %f %f %f %f", step, voltage, current, OVP, time);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
    
}

/*===========================================================================*/
/* Function: ProgVoltRamp                                              		 */
/* Purpose:  This routine programs the selected step for a Voltage Ramp at	 */
/*			 the defined Voltage, Current, and OVP values for the specified  */
/*			 time. The voltage will ramp from the present voltage setting to */
/*			 the value programmed in the voltage control.					 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ProgVoltRamp (ViSession instrSession, ViInt16 step,
                             ViReal64 voltage,ViReal64 voltage_final, ViReal64 current, ViReal64 OVP,
                             ViReal64 time)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"PROGRAM:DEFINE ");
    sprintf(tmpBuffer,"%d RAMPTOV %f %f %f %f %f", step, voltage, voltage_final, current, OVP, time);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
    
}

/*===========================================================================*/
/* Function: ProgCurrRamp                                             		 */
/* Purpose:  This routine programs the selected step for a Current Ramp at	 */
/*			 the defined Voltage, Current, and OVP values for the specified	 */
/*			 time. The current will ramp from the present current setting to */
/*			 the value programmed in the current control. An appropriate load*/
/*			 is required for current mode operation.						 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ProgCurrRamp (ViSession instrSession, ViInt16 step,
                             ViReal64 voltage, ViReal64 current,ViReal64 current_final, ViReal64 OVP,
                             ViReal64 time)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"PROGRAM:DEFINE ");
    sprintf(tmpBuffer,"%d RAMPTOC %f %f %f %f %f", step, voltage, current, current_final, OVP, time);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
    
}

/*===========================================================================*/
/* Function: ProgPowerMode                                             		 */
/* Purpose:  This routine programs the selected step for a Constant Power at */
/*			 the defined Voltage, Current, and OVP values for the specified  */
/*			 time. If the specified power setting can not be realized then	 */
/*			 the power supply output will be limited to the voltage or		 */
/*			 current setting. These conditions are load dependent.			 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ProgPowerMode (ViSession instrSession, ViInt16 step, ViReal64 power,
                              ViReal64 voltage, ViReal64 current, ViReal64 OVP,
                              ViReal64 time)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[90];
    ViByte		tmpBuffer[90];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"PROGRAM:DEFINE ");
    sprintf(tmpBuffer,"%d POWERSETTINGS %f %f %f %f %f", step, power, voltage, current, OVP, time);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
    
}

/*===========================================================================*/
/* Function: ProgRepeat                                                		 */
/* Purpose:  This routine programs the defined sequence to repeat 			 */
/*			 continuously until a Pause or Stop command is issued..			 */
/*																			 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ProgRepeat (ViSession instrSession, ViInt16 step)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"PROGRAM:DEFINE ");
    sprintf(tmpBuffer,"%d REPEAT", step);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
    
}

/*===========================================================================*/
/* Function: ProgCall                                                		 */
/* Purpose:  This routine programs the presently defined sequence to begin	 */
/*			 execution of the specified sequence. See the Sequence Return	 */
/*			 command.														 */
/*																			 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ProgCall (ViSession instrSession, ViInt16 step,
                         ViChar sequence[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"PROGRAM:DEFINE ");
    sprintf(tmpBuffer,"%d SUBCALL \"%s\"", step, sequence);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
    
}

/*===========================================================================*/
/* Function: ProgReturn                                                		 */
/* Purpose:  This routine returns sequence execution to the next step of the */
/*			 calling sequence in which the Call command was defined.		 */
/*																			 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ProgReturn (ViSession instrSession, ViInt16 step)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"PROGRAM:DEFINE ");
    sprintf(tmpBuffer,"%d RETURN", step);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
    
}

/*===========================================================================*/
/* Function: ProgLoop                                                		 */
/* Purpose:  This routine programs a sequence loop command. The set of		 */
/*			 sequence steps between the Loop command and the following Next	 */
/*			 command will be repeditively executed for the count specified.	 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ProgLoop (ViSession instrSession, ViInt16 step,
                         unsigned int loop)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"PROGRAM:DEFINE ");
    sprintf(tmpBuffer,"%d LOOP %d", step, loop);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
    
}

/*===========================================================================*/
/* Function: ProgNext                                                		 */
/* Purpose:  This routine programs a sequence next command. See Sequence Loop*/
/*			 command.														 */
/*																			 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ProgNext (ViSession instrSession, ViInt16 step)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"PROGRAM:DEFINE ");
    sprintf(tmpBuffer,"%d NEXT", step);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
    
}

/*===========================================================================*/
/* Function: ProgStop                                                		 */
/* Purpose:  This routine programs a sequence stop command which halts		 */
/*			 sequence execution.											 */
/*																			 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ProgStop (ViSession instrSession, ViInt16 step)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"PROGRAM:DEFINE ");
    sprintf(tmpBuffer,"%d STOP", step);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
    
}

/*===========================================================================*/
/* Function: ProgGoto                                                		 */
/* Purpose:  This routine programs a sequence goto command. Sequence		 */
/*			 execution will begin at the specified Sequence Name.			 */
/*																			 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ProgGoto (ViSession instrSession, ViInt16 step,
                         ViChar sequence[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"PROGRAM:DEFINE ");
    sprintf(tmpBuffer,"%d GOTO \"%s\"", step, sequence);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
    
}

/*===========================================================================*/
/* Function: ProgPause                                                		 */
/* Purpose:  This routine programs a sequence pause command which will stop	 */
/*			 sequence execution until a Sequence Resume command is sent.	 */
/*																			 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ProgPause (ViSession instrSession, ViInt16 step)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"PROGRAM:DEFINE ");
    sprintf(tmpBuffer,"%d PAUSE", step);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
    
}


/*===========================================================================*/
/* Function: ProgNOP                                                		 */
/* Purpose:  This routine programs a sequence NOP command which will 	 */
/*			 have no effect on the sequence. The sequence will	 */
/*			 continue onto the next step.									 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ProgNOP (ViSession instrSession, ViInt16 step)
{
      ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"PROGRAM:DEFINE ");
    sprintf(tmpBuffer,"%d NOP", step);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK   
}


/*===========================================================================*/
/* Function: ProgDefineQuery                                           		 */
/* Purpose:  This routine returns the V,I,OVP,... settings for the specified */
/*			 step of the presently selected sequence.						 */
/*																			 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ProgDefineQuery (ViSession instrSession, ViInt16 step,
                                ViChar settings[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];        
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string - voltage limit */
    strcpy(cmdBuffer,"PROGRAM:DEFINE? ");
    sprintf(tmpBuffer,"%d",step);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* get returned query string */
    if ((SNSGI_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    strcpy(settings, rdBuffer);

    UNLOCK
    
}

/*===========================================================================*/
/* Function: ProgDeleteSelected                                        		 */
/* Purpose:  This routine deletes the selected sequence.					 */
/*																			 */
/*																			 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ProgDeleteSelected (ViSession instrSession)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"PROGRAM:DELETE:SELECTED");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
    
}

/*===========================================================================*/
/* Function: ProgDeleteAll                                                	 */
/* Purpose:  This routine deletes all sequences.							 */
/*																			 */
/*																			 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ProgDeleteAll (ViSession instrSession)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"PROGRAM:DELETE:ALL");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
    
}

/*===========================================================================*/
/* Function: ImmExeVI                                                		 */
/* Purpose:  This routine programs and executes a VIMODE step immediately.	 */
/*			 The output will be programmed to the specified values.			 */
/*																			 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ImmExeVI (ViSession instrSession, ViReal64 voltage,
                         ViReal64 current, ViReal64 OVP)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"PROGRAM:EXECUTE VIMODE ");
    sprintf(tmpBuffer,"%f %f %f",voltage, current, OVP);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
    
}

/*===========================================================================*/
/* Function: ImmExeVoltRamp                                            		 */
/* Purpose:  This routine programs and immediately executes a Voltage Ramp at*/
/*			 the defined Voltage, Current, and OVP values for the specified	 */
/*			 time. The voltage will ramp from the present voltage setting to */
/*			 the value programmed in the voltage control.					 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ImmExeVoltRamp (ViSession instrSession, ViReal64 voltage, ViReal64 voltage_final,
                               ViReal64 current, ViReal64 OVP, ViReal64 time)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"PROGRAM:EXECUTE RAMPTOV ");
    sprintf(tmpBuffer,"%f %f %f %f %f",voltage,voltage_final, current, OVP, time);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
    
}

/*===========================================================================*/
/* Function: ImmExeCurrRamp                                            		 */
/* Purpose:  This routine programs and executes a Current Ramp at the defined*/
/*			 Voltage, Current, and OVP values for the specified time. The	 */
/*			 current will ramp from the present current setting to the value */
/*			 programmed in the current control. An appropriate load is		 */
/*			 required for current mode operation.							 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ImmExeCurrRamp (ViSession instrSession, ViReal64 voltage,
                               ViReal64 current,ViReal64 current_final, ViReal64 OVP,
                               ViReal64 time)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"PROGRAM:EXECUTE RAMPTOC ");
    sprintf(tmpBuffer,"%f %f %f %f %f",voltage, current,current_final, OVP, time);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
    
}

/*===========================================================================*/
/* Function: ImmExePowerMode                                           		 */
/* Purpose:  This routine programs and immediately executes Constant Power	 */
/*			 Mode at the defined Power, Voltage, Current, and OVP values. If */
/*			 the specified power setting can not be realized then the power	 */
/*			 supply output will be limited to the voltage or current setting.*/
/*			 These conditions are load dependent.							 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ImmExePowerMode (ViSession instrSession, ViReal64 voltage,
                                ViReal64 current, ViReal64 OVP, ViReal64 power)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"PROGRAM:EXECUTE POWERSETTINGS ");
    sprintf(tmpBuffer,"%f %f %f %f",power,voltage, current, OVP);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
    
}

/*===========================================================================*/
/* Function: ProgMemAlloc                                             		 */
/* Purpose:  This function allocates memory for a newly named sequence.		 */
/*																			 */
/*																			 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ProgMemAlloc (ViSession instrSession)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"PROGRAM:MALLOCATE DEFAULT");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
    
}

/*===========================================================================*/
/* Function: ProgName                                                		 */
/* Purpose:  This function selects an already defined sequence for use or	 */
/*			 creates a new sequence if the name does not exist.				 */
/*																			 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ProgName (ViSession instrSession, ViChar name[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];
    
	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"PROGRAM:NAME ");
    sprintf(tmpBuffer,"\"%s\"",name);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);

    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK

}

/*===========================================================================*/
/* Function: ProgNameChan                                                		 */
/* Purpose:  This function selects an already defined sequence for use or	 */
/*			 creates a new sequence if the name does not exist.  			 */
/*			 A number after the name indicates which channel to use.		 */
/*																			 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ProgNameChan (ViSession instrSession, ViChar name[], int chan)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];
    
    
	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"PROGRAM:NAME ");
    sprintf(tmpBuffer,"\"%s\", %d",name,chan);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);

    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
    
}

/*===========================================================================*/
/* Function: ProgNameQuery                                             		 */
/* Purpose:  This function returns the name of the active sequence. If no	 */
/*			 sequence have been defined then a default name is returned.	 */
/*																			 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ProgNameQuery (ViSession instrSession, ViChar name[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string - voltage limit */
    strcpy(cmdBuffer,"PROGRAM:NAME?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* get returned query string */
    if ((SNSGI_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    strcpy(name, rdBuffer);

    UNLOCK
    
}
/*===========================================================================*/
/* Function: ProgSaveSelected                                        		 */
/* Purpose:  This routine saves the selected sequence.				    	 */
/*																	 		 */
/*																			 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ProgSaveSelected(ViSession instrSession)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"PROGRAM:SAVE:SELECTED");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
    
}
/*===========================================================================*/
/* Function: ProgSaveAll                                                	 */
/* Purpose:  This routine saves all sequences.						    	 */
/*																			 */
/*																			 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ProgSaveAll (ViSession instrSession)
{
        ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"PROGRAM:SAVE:ALL");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
}

/*===========================================================================*/
/* Function: ProgState                                                		 */
/* Purpose:  This function sets the sequence state to the specified value.	 */
/*			 Valid states are:												 */
/*																			 */
/*			 Run														 */
/*			 Pause														 */
/*			 Stop														 */
/*			 Resume														 */
/*			 Complete													 */
/*																			 */
/*			 Note: Complete should be issued to terminate an editing session.*/
/*			 This would be used to save a sequence prior to defining another */
/*			 sequence.														 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ProgState (ViSession instrSession, ViChar name[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"PROGRAM:STATE ");
    sprintf(tmpBuffer, "%s",name);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
    
}

/*===========================================================================*/
/* Function: ProgStateQuery                                            		 */
/* Purpose:  This function returns the sequence state.						 */
/*																			 */
/*			 Valid states are:												 */
/*																			 */
/*			 0 = Run														 */
/*			 1 = Pause														 */
/*			 2 = Stop														 */
/*			 3 = Resume														 */
/*			 4 = Complete													 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ProgStateQuery (ViSession instrSession, ViInt16 *state)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string - voltage limit */
    strcpy(cmdBuffer,"PROGRAM:STATE?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* get returned query string */
    if ((SNSGI_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *state = atof(rdBuffer);

    UNLOCK
    
}

/*===========================================================================*/
/* Function: SeqWait                                                		 */
/* Purpose:  This routine gives a means for the users GPIB system computer to*/
/*			 know when a sequence has completed.							 */
/*																			 */
/*			 NOTE:	Currently not used										 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_SeqWait (ViSession instrSession, ViInt16 step,
                        ViChar sequence[])
{
    ViStatus	SNSGI_status,unlockStatus;

	LOCK
	UNLOCK
    
}

/*===========================================================================*/
/* Function: SeqTrig                                                		 */
/* Purpose:  This routine triggers the selected sequence than has been		 */
/*			 priviously armed.												 */
/*																			 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_SeqTrig (ViSession instrSession)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"PROGRAM:TRIGGER");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
    
}


/*===========================================================================*/
/* Function: SendSWTrigger                                                   */
/* Purpose:  This function sends a software trigger to the specified         */
/*           channel. Setting the channel to zero causes a Group Execute     */
/*           Trigger for all power supplies that have been previously        */
/*           configured for triggered outputs.                               */
/*           Trigger Types: 1 = Voltage, 2 = Current, 3 = Both, 4 = Ramp     */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_SendSWTrigger (ViSession      instrSession,
                                      ViInt16        type)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK
	
	/* test for voltage, current, or both trigger type */
    if ((type > 0) && (type < 4))
    {
		/* build SCPI command string */
	    strcpy(cmdBuffer,"TRIGGER:TYPE ");
	    sprintf(tmpBuffer,"%d",type);
	    strcat(cmdBuffer,tmpBuffer);
	}
	/* test for ramp trigger type */
	else if (type == 4)
    {
		/* build SCPI command string */
	    strcpy(cmdBuffer,"TRIGGER:RAMP");
	}
	else
		return VI_ERROR_INV_PARAMETER;
	
	/* send SCPI command string */
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    UNLOCK
}

/*===========================================================================*/
/* Function: Abort Trigger                                                   */
/* Purpose:  This function clears all previously configured voltage and      */
/*           current triggers for the specified channel.                     */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_AbortTrigger (ViSession      instrSession)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];

	LOCK
	
	/* build and send SCPI command string */
    strcpy(cmdBuffer,"TRIGGER:ABORT");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: StatusByteQuery                                                 */
/* Purpose:  This function returns the instruments status byte and a message */
/*           indicating which bits are active.                               */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_StatusByteQuery (ViSession      instrSession,
                                        unsigned char  *statusByte,
                                        ViChar _VI_FAR statusMessage[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViChar		rdBuffer[BUFFER_SIZE];
    char		*endPtr;

	LOCK
	
	/* build and send SCPI command string */
    strcpy(cmdBuffer,"*STB?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query string */
    if ((SNSGI_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* assign value */
	*statusByte = (unsigned char) strtoul((const char *) rdBuffer, &endPtr, 10);

	/* generate status message from status byte bits, bits 0,3,7 are NOT USED */
	if (*statusByte == 0)
		strcpy(statusMessage,"00");
	else
	{
		sprintf(rdBuffer,"%02X - ",*statusByte);
		if (*statusByte & SNSGI_SB_PROTEVENT)
			strcat(rdBuffer,"Protection Event, ");
		if (*statusByte & SNSGI_SB_ERRORQUERY)
			strcat(rdBuffer,"Error Event Message, ");
		if (*statusByte & SNSGI_SB_MAV)
			strcat(rdBuffer,"Message Available, ");
		if (*statusByte & SNSGI_SB_STANDEVENT)
			strcat(rdBuffer,"Event Status, ");
		if (*statusByte & SNSGI_SB_SRQ)
			strcat(rdBuffer,"Request Service");
		
		/* remove any ending ',' */
		retCnt = strlen(rdBuffer);
		if (rdBuffer[retCnt - 2] == ',')
			rdBuffer[retCnt - 2] = '\0';

		strcpy(statusMessage,rdBuffer);
	}
	
    UNLOCK
}

/*===========================================================================*/
/* Function: StandEventQuery                                                 */
/* Purpose:  This function returns the instruments standard event status     */
/*           register and a message indicating which bits are active.        */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_StandEventQuery (ViSession      instrSession,
                                        unsigned char  *standardEventStatusRegister,
                                        ViChar _VI_FAR statusMessage[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViChar		rdBuffer[BUFFER_SIZE];
    char		*endPtr;

	LOCK
	
	/* build and send SCPI command string */
    strcpy(cmdBuffer,"*ESR?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query string */
    if ((SNSGI_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* assign value */
	*standardEventStatusRegister = (unsigned char) strtoul((const char *) rdBuffer, &endPtr, 10);

	/* generate status message from register bits, bits 1,6 are NOT USED */
	if (*standardEventStatusRegister == 0)
		strcpy(statusMessage,"00");
	else
	{
		sprintf(rdBuffer,"%02X - ",*standardEventStatusRegister);
		if (*standardEventStatusRegister & SNSGI_SE_OPC)
			strcat(rdBuffer,"Operation Complete, ");
		if (*standardEventStatusRegister & SNSGI_SE_ERRORQUERY)
			strcat(rdBuffer,"Query Error, ");
		if (*standardEventStatusRegister & SNSGI_SE_DEVICEERROR)
			strcat(rdBuffer,"Device Error, ");
		if (*standardEventStatusRegister & SNSGI_SE_RANGEERROR)
			strcat(rdBuffer,"Execution Error (Range), ");
		if (*standardEventStatusRegister & SNSGI_SE_SYNTAXERROR)
			strcat(rdBuffer,"Command Error (Syntax), ");
		if (*standardEventStatusRegister & SNSGI_SE_PWRON)
			strcat(rdBuffer,"Power On");
		
		/* remove any ending ',' */
		retCnt = strlen(rdBuffer);
		if (rdBuffer[retCnt - 2] == ',')
			rdBuffer[retCnt - 2] = '\0';

		strcpy(statusMessage,rdBuffer);
	}
	
    UNLOCK
}

/*===========================================================================*/
/* Function: StandEventEnableQuery                                           */
/* Purpose:  This function returns the instruments standard event enable     */
/*           register and a message indicating which bits are active.        */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_StandEventEnableQuery (ViSession      instrSession,
                                              unsigned char  *standardEventEnableRegister,
                                              ViChar _VI_FAR statusMessage[])
{
    
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViChar		rdBuffer[BUFFER_SIZE];
    char		*endPtr;

	LOCK
	
	/* build and send SCPI command string */
    strcpy(cmdBuffer,"*ESE?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query string */
    if ((SNSGI_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* assign value */
	*standardEventEnableRegister = (unsigned char) strtoul((const char *) rdBuffer, &endPtr, 10);

	/* generate status message from register bits, bits 1,6 are NOT USED */
	if (*standardEventEnableRegister == 0)
		strcpy(statusMessage,"00");
	else
	{
		sprintf(rdBuffer,"%02X - ",*standardEventEnableRegister);
		if (*standardEventEnableRegister & SNSGI_SE_OPC)
			strcat(rdBuffer,"Operation Complete, ");
		if (*standardEventEnableRegister & SNSGI_SE_ERRORQUERY)
			strcat(rdBuffer,"Query Error, ");
		if (*standardEventEnableRegister & SNSGI_SE_DEVICEERROR)
			strcat(rdBuffer,"Device Error, ");
		if (*standardEventEnableRegister & SNSGI_SE_RANGEERROR)
			strcat(rdBuffer,"Execution Error (Range), ");
		if (*standardEventEnableRegister & SNSGI_SE_SYNTAXERROR)
			strcat(rdBuffer,"Command Error (Syntax), ");
		if (*standardEventEnableRegister & SNSGI_SE_PWRON)
			strcat(rdBuffer,"Power On");
		
		/* remove any ending ',' */
		retCnt = strlen(rdBuffer);
		if (rdBuffer[retCnt - 2] == ',')
			rdBuffer[retCnt - 2] = '\0';

		strcpy(statusMessage,rdBuffer);
	}
	
    UNLOCK
}

/*===========================================================================*/
/* Function: ProtEventQuery                                                  */
/* Purpose:  This function returns the instruments protection event status   */
/*           register and a message indicating which bits are active.        */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ProtEventQuery (ViSession      instrSession,
                                       unsigned char  *protEventReg,
                                       ViChar _VI_FAR statusMessage[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViChar		rdBuffer[BUFFER_SIZE];
    char		*endPtr;

	LOCK
	
	/* build and send SCPI command string */
    strcpy(cmdBuffer,"STATUS:PROTECTION:EVENT?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query string */
    if ((SNSGI_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* assign value */
	*protEventReg = (unsigned char) strtoul((const char *) rdBuffer, &endPtr, 10);

	/* generate status message from register bits */
	if (*protEventReg == 0)
		strcpy(statusMessage,"00");
	else
	{
		sprintf(rdBuffer,"%02X - ",*protEventReg);
		if (*protEventReg & SNSGI_PE_VOLTMODE)
			strcat(rdBuffer,"Constant Voltage, ");
		if (*protEventReg & SNSGI_PE_CURRMODE)
			strcat(rdBuffer,"Constant Current, ");
		if (*protEventReg & SNSGI_PE_HW)
			strcat(rdBuffer,"Hardware Fault, ");
		if (*protEventReg & SNSGI_PE_OVP)
			strcat(rdBuffer,"OVP, ");
		if (*protEventReg & SNSGI_PE_OTP)
			strcat(rdBuffer,"OTP, ");
		if (*protEventReg & SNSGI_PE_EXTSD)
			strcat(rdBuffer,"Ext. Shutdown, ");
		if (*protEventReg & SNSGI_PE_FOLD)
			strcat(rdBuffer,"Foldback Mode, ");
		if (*protEventReg & SNSGI_PE_PRGMERROR)
			strcat(rdBuffer,"Programming Error");
		
		/* remove any ending ',' */
		retCnt = strlen(rdBuffer);
		if (rdBuffer[retCnt - 2] == ',')
			rdBuffer[retCnt - 2] = '\0';

		strcpy(statusMessage,rdBuffer);
	}
	
    UNLOCK
}

/*===========================================================================*/
/* Function: ProtEventEnableQuery                                            */
/* Purpose:  This function returns the instruments protection event enable   */
/*           register and a message indicating which bits are active.        */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ProtEventEnableQuery (ViSession      instrSession,
                                             unsigned char  *protEventEnableRegister,
                                             ViChar _VI_FAR statusMessage[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViChar		rdBuffer[BUFFER_SIZE];
    char		*endPtr;

	LOCK
	
	/* build and send SCPI command string */
    strcpy(cmdBuffer,"STATUS:PROTECTION:ENABLE?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query string */
    if ((SNSGI_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* assign value */
	*protEventEnableRegister = (unsigned char) strtoul((const char *) rdBuffer, &endPtr, 10);

	/* generate status message from register bits */
	if (*protEventEnableRegister == 0)
		strcpy(statusMessage,"00");
	else
	{
		sprintf(rdBuffer,"%02X - ",*protEventEnableRegister);
		if (*protEventEnableRegister & SNSGI_PE_VOLTMODE)
			strcat(rdBuffer,"Constant Voltage, ");
		if (*protEventEnableRegister & SNSGI_PE_CURRMODE)
			strcat(rdBuffer,"Constant Current, ");
		if (*protEventEnableRegister & SNSGI_PE_HW)
			strcat(rdBuffer,"Hardware Fault, ");
		if (*protEventEnableRegister & SNSGI_PE_OVP)
			strcat(rdBuffer,"OVP, ");
		if (*protEventEnableRegister & SNSGI_PE_OTP)
			strcat(rdBuffer,"OTP, ");
		if (*protEventEnableRegister & SNSGI_PE_EXTSD)
			strcat(rdBuffer,"Ext. Shutdown, ");
		if (*protEventEnableRegister & SNSGI_PE_FOLD)
			strcat(rdBuffer,"Foldback Mode, ");
		if (*protEventEnableRegister & SNSGI_PE_PRGMERROR)
			strcat(rdBuffer,"Programming Error");
		
		/* remove any ending ',' */
		retCnt = strlen(rdBuffer);
		if (rdBuffer[retCnt - 2] == ',')
			rdBuffer[retCnt - 2] = '\0';

		strcpy(statusMessage,rdBuffer);
	}
	
    UNLOCK
}

/*===========================================================================*/
/* Function: ProtCondQuery                                                   */
/* Purpose:  This function returns the instruments protection condition      */
/*           status register and a message indicating which bits are active. */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ProtCondQuery (ViSession      instrSession,
                                      unsigned char  *protCondReg,
                                      ViChar _VI_FAR statusMessage[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViChar		rdBuffer[BUFFER_SIZE];
    char		*endPtr;

	LOCK
	
	/* build and send SCPI command string */
    strcpy(cmdBuffer,"STATUS:PROTECTION:CONDITION?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query string */
    if ((SNSGI_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* assign value */
	*protCondReg = (unsigned char) strtoul((const char *) rdBuffer, &endPtr, 10);

	/* generate status message from register bits */
	if (*protCondReg == 0)
		strcpy(statusMessage,"00");
	else
	{
		sprintf(rdBuffer,"%02X - ",*protCondReg);
		if (*protCondReg & SNSGI_PE_VOLTMODE)
			strcat(rdBuffer,"Constant Voltage, ");
		if (*protCondReg & SNSGI_PE_CURRMODE)
			strcat(rdBuffer,"Constant Current, ");
		if (*protCondReg & SNSGI_PE_HW)
			strcat(rdBuffer,"Hardware Fault, ");
		if (*protCondReg & SNSGI_PE_OVP)
			strcat(rdBuffer,"OVP, ");
		if (*protCondReg & SNSGI_PE_OTP)
			strcat(rdBuffer,"OTP, ");
		if (*protCondReg & SNSGI_PE_EXTSD)
			strcat(rdBuffer,"Ext. Shutdown, ");
		if (*protCondReg & SNSGI_PE_FOLD)
			strcat(rdBuffer,"Foldback Mode, ");
		if (*protCondReg & SNSGI_PE_PRGMERROR)
			strcat(rdBuffer,"Programming Error");
		
		/* remove any ending ',' */
		retCnt = strlen(rdBuffer);
		if (rdBuffer[retCnt - 2] == ',')
			rdBuffer[retCnt - 2] = '\0';

		strcpy(statusMessage,rdBuffer);
	}
	
    UNLOCK
}

/*===========================================================================*/
/* Function: ServiceReqEnableQuery                                           */
/* Purpose:  This function returns the instruments service request enable    */
/*           register and a message indicating which bits are active.        */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ServiceReqEnableQuery (ViSession      instrSession,
                                              unsigned char  *serviceReqEnableRegister,
                                              ViChar _VI_FAR statusMessage[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViChar		rdBuffer[BUFFER_SIZE];
    char		*endPtr;

	LOCK
	
	/* build and send SCPI command string */
    strcpy(cmdBuffer,"*SRE?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query string */
    if ((SNSGI_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* assign value */
	*serviceReqEnableRegister = (unsigned char) strtoul((const char *) rdBuffer, &endPtr, 10);

	/* generate status message from status byte bits, bits 0,3,7 are NOT USED */
	if (*serviceReqEnableRegister == 0)
		strcpy(statusMessage,"00");
	else
	{
		sprintf(rdBuffer,"%02X - ",*serviceReqEnableRegister);
		if (*serviceReqEnableRegister & 0x02)
			strcat(rdBuffer,"Protection Event, ");
		if (*serviceReqEnableRegister & 0x04)
			strcat(rdBuffer,"Error Event Message, ");
		if (*serviceReqEnableRegister & 0x10)
			strcat(rdBuffer,"Message Available, ");
		if (*serviceReqEnableRegister & 0x20)
			strcat(rdBuffer,"Event Status, ");
		if (*serviceReqEnableRegister & 0x40)
			strcat(rdBuffer,"Request Service");
		
		/* remove any ending ',' */
		retCnt = strlen(rdBuffer);
		if (rdBuffer[retCnt - 2] == ',')
			rdBuffer[retCnt - 2] = '\0';

		strcpy(statusMessage,rdBuffer);
	}
	
    UNLOCK
}

/*===========================================================================*/
/* Function: Reset                                                           */
/* Purpose:  This function resets the instrument to a known state.           */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_reset (ViSession      instrSession)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];

	LOCK
	
    /*  send SCPI reset command */
    strcpy(cmdBuffer,"*RST");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    UNLOCK
}

/*===========================================================================*/
/* Function: Clear                                                          */
/* Purpose:  This function clear the instrument error and status info.       */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_clear (ViSession      instrSession)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];

	LOCK
	
	/* send SCPI clear command */
    strcpy(cmdBuffer,"*CLS");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: Error Query                                                     */
/* Purpose:  This function queries the instrument error queue, and returns   */
/*           the result.                                                     */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_errorQuery (ViSession      instrSession,
                                   ViPInt32       errorCode,
                                   ViChar _VI_FAR errorMessage[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViUInt32	index;
    ViUInt32	i;
    ViStatus	SNSGI_status,unlockStatus;
    ViChar		*cptr;
    ViChar		rdBuffer[BUFFER_SIZE];
    ViByte		cmdBuffer[80];

	LOCK
	
    /*  send SCPI reset command */
    strcpy(cmdBuffer,"SYSTEM:ERROR?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

	/* get returned query string */
    if ((SNSGI_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* get error code from string */
	sscanf(rdBuffer,"%d",errorCode);
	
	/* get error message, skip ", " in return string */
	cptr  = strchr(rdBuffer,',');
	cptr += 2;
	strcpy(errorMessage,cptr);

	UNLOCK
}

/*===========================================================================*/
/* Function: Error Message                                                   */
/* Purpose:  This function translates the error return value from the        */
/*           instrument driver into a user-readable string.                  */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_errorMessage (ViSession      instrSession,
                                     ViStatus       statusCode,
                                     ViChar _VI_FAR message[])
{
    ViStatus SNSGI_status,unlockStatus;
    ViInt16 i;
    static SNSGI_tStringValPair statusDescArray[] = {
        {VI_WARN_NSUP_ID_QUERY,                "WARNING: ID Query not supported"},
        {VI_WARN_NSUP_RESET,                   "WARNING: Reset not supported"},
        {VI_WARN_NSUP_SELF_TEST,               "WARNING: Self-test not supported"},
        {VI_WARN_NSUP_ERROR_QUERY,             "WARNING: Error Query not supported"},     
        {VI_WARN_NSUP_REV_QUERY,               "WARNING: Revision Query not supported"},
        {VI_ERROR_PARAMETER1,                  "ERROR: Parameter 1 out of range"},
        {VI_ERROR_PARAMETER2,                  "ERROR: Parameter 2 out of range"},
        {VI_ERROR_PARAMETER3,                  "ERROR: Parameter 3 out of range"},
        {VI_ERROR_PARAMETER4,                  "ERROR: Parameter 4 out of range"},
        {VI_ERROR_PARAMETER5,                  "ERROR: Parameter 5 out of range"},
        {VI_ERROR_PARAMETER6,                  "ERROR: Parameter 6 out of range"},
        {VI_ERROR_PARAMETER7,                  "ERROR: Parameter 7 out of range"},
        {VI_ERROR_PARAMETER8,                  "ERROR: Parameter 8 out of range"},
        {VI_ERROR_FAIL_ID_QUERY,               "ERROR: Identification query failed"},
        {VI_ERROR_INV_RESPONSE,                "ERROR: Interpreting instrument response"},
        {VI_ERROR_INSTR_FILE_OPEN,             "ERROR: Opening the specified file"},
        {VI_ERROR_INSTR_FILE_WRITE,            "ERROR: Writing to the specified file"},
        {VI_ERROR_INSTR_INTERPRETING_RESPONSE, "ERROR: Interpreting the instrument's response"},
        
        /* not used, example of user defined Status Code
        {PREFIX_ERROR_INVALID_CONFIGURATION,   "ERROR: Instrument configuration error"},
        */
        
        {VI_NULL, VI_NULL}
    };

    SNSGI_status = viStatusDesc (instrSession, statusCode, message);
    if (SNSGI_status == VI_WARN_UNKNOWN_STATUS) {
        for (i=0; statusDescArray[i].stringName; i++) {
            if (statusDescArray[i].stringVal == statusCode) {
                strcpy (message, statusDescArray[i].stringName);
                return (VI_SUCCESS);
            }
        }
        sprintf (message, "Unknown Error 0x%08lX", statusCode);
        return (VI_WARN_UNKNOWN_STATUS);
    }
    
    UNLOCK
}

/*===========================================================================*/
/* Function: revisionQuery                                                   */
/* Purpose:  This function returns the driver and instrument revisions.      */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_revisionQuery (ViSession      instrSession,
                                      ViChar _VI_FAR driverRev[],
                                      ViChar _VI_FAR instrRev[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViInt16		i;
    ViChar		*cptr;
    ViByte		rdBuffer[BUFFER_SIZE];
    ViByte		cmdBuffer[BUFFER_SIZE];

	LOCK
	
	/* send SCPI identify command */
	strcpy(cmdBuffer,"*IDN?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
	sendCnt = strlen(cmdBuffer);
    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query */
    if ((SNSGI_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

    /* parse *idn? return string for firware revisions - 4 & 5 fields */
	cptr = rdBuffer;
	for (i = 0; i < 3; ++i)
	{
		cptr = strchr(cptr,',');
		++cptr;
	}
	
	/* copy driver and intrument strings to buffers */
	strcpy(instrRev, cptr);
    strcpy (driverRev, SNSGI_REVISION);

	UNLOCK
}

/*===========================================================================*/
/* Function: Identify Query                                                  */
/* Purpose:  This function returns the *idn? response.                       */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_idQuery (ViSession      instrSession,
                                ViChar _VI_FAR idBuffer[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViInt16		i;
    ViChar		*cptr;
    ViByte		rdBuffer[BUFFER_SIZE];
    ViByte		cmdBuffer[80];

	LOCK
	
	/* send SCPI identify command */
	strcpy(cmdBuffer,"*IDN?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
	sendCnt = strlen(cmdBuffer);
    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query */
    if ((SNSGI_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* copy id string to buffer */
	strcpy(idBuffer, rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: ModelQuery                                                   */
/* Purpose:  This function reads the instrument's Maximum Voltage and        */
/*           Current for the specified channel.                              */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_ModelQuery    (ViSession      instrSession,
                                      ViReal64       *modelVolt,
                                      ViReal64       *modelCurr)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGI_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    *modelVolt = 0.0;
    *modelCurr = 0.0;

    /* build and send SCPI command string - voltage */
    strcpy(cmdBuffer,"CALIBRATE:MODEL:VOLTAGE?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query */
    if ((SNSGI_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *modelVolt = atof(rdBuffer);

    /* build and send SCPI command string - current */
    strcpy(cmdBuffer,"CALIBRATE:MODEL:CURRENT?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGI_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query */
    if ((SNSGI_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *modelCurr = atof(rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: Close                                                           */
/* Purpose:  This function closes the instrument.                            */
/*===========================================================================*/
ViStatus CVIFUNC SNSGI_close (ViSession instrSession)
{
    SNSGI_instrRange instrPtr;
    ViSession rmSession;
    ViStatus SNSGI_status;

    if ((SNSGI_status = viGetAttribute (instrSession, VI_ATTR_RM_SESSION, &rmSession)) < 0)
		return SNSGI_status;
 	if ((SNSGI_status = viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr)) < 0)
		return SNSGI_status;
            
    if (instrPtr != NULL) 
        free (instrPtr);
    
    SNSGI_status = viClose (instrSession);
    viClose (rmSession);

	return SNSGI_status;
}

/*****************************************************************************/
/*= UTILITY ROUTINES (Non-Exportable Functions) =============================*/
/*****************************************************************************/

/*===========================================================================*/
/* Function: Boolean Value Out Of Range - ViBoolean                          */
/* Purpose:  This function checks a Boolean to see if it is equal to VI_TRUE */
/*           or VI_FALSE. If the value is out of range, the return value is  */
/*           VI_TRUE, otherwise the return value is VI_FALSE.                */
/*===========================================================================*/
ViBoolean SNSGI_invalidViBooleanRange (ViBoolean val)
{
    return ((val != VI_FALSE && val != VI_TRUE) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Short Signed Integer Value Out Of Range - ViInt16               */
/* Purpose:  This function checks a short signed integer value to see if it  */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range, the return value is VI_TRUE, otherwise the return     */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean SNSGI_invalidViInt16Range (ViInt16 val, ViInt16 min, ViInt16 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Long Signed Integer Value Out Of Range - ViInt32                */
/* Purpose:  This function checks a long signed integer value to see if it   */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range, the return value is VI_TRUE, otherwise the return     */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean SNSGI_invalidViInt32Range (ViInt32 val, ViInt32 min, ViInt32 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Unsigned Char Value Out Of Range - ViUInt8                      */
/* Purpose:  This function checks an unsigned char value to see if it        */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range, the return value is VI_TRUE, otherwise the return     */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean SNSGI_invalidViUInt8Range (ViUInt8 val, ViUInt8 min, ViUInt8 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Short Unsigned Integer Value Out Of Range - ViUInt16            */
/* Purpose:  This function checks a short unsigned integer value to see if it*/  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range, the return value is VI_TRUE, otherwise the return     */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean SNSGI_invalidViUInt16Range (ViUInt16 val, ViUInt16 min, ViUInt16 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Long Unsigned Integer Value Out Of Range - ViUInt32             */
/* Purpose:  This function checks a long unsigned integer value to see if it */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range, the return value is VI_TRUE, otherwise the return     */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean SNSGI_invalidViUInt32Range (ViUInt32 val, ViUInt32 min, ViUInt32 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Real (Float) Value Out Of Range - ViReal32                      */
/* Purpose:  This function checks a real (float) value to see if it lies     */  
/*           between a minimum and maximum value.  If the value is out of    */
/*           range, the return value is VI_TRUE, otherwise the return value  */
/*           is VI_FALSE.                                                    */
/*===========================================================================*/
ViBoolean SNSGI_invalidViReal32Range (ViReal32 val, ViReal32 min, ViReal32 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Real (Double) Value Out Of Range - ViReal64                     */
/* Purpose:  This function checks a real (double) value to see if it lies    */  
/*           between a minimum and maximum value.  If the value is out of    */
/*           range, the return value is VI_TRUE, otherwise the return value  */
/*           is VI_FALSE.                                                    */
/*===========================================================================*/
ViBoolean SNSGI_invalidViReal64Range (ViReal64 val, ViReal64 min, ViReal64 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Initialize Clean Up                                             */
/* Purpose:  This function is used only by the SNSGI_init function.  When   */
/*           an error is detected this function is called to close the       */
/*           open resource manager and instrument object sessions and to     */
/*           set the instrSession that is returned from SNSGI_init to       */
/*           VI_NULL.                                                        */
/*===========================================================================*/
ViStatus SNSGI_initCleanUp (ViSession  openRMSession,
                            ViPSession openInstrSession,
                            ViStatus   currentStatus)
{
    viClose (*openInstrSession);
    viClose (openRMSession);
    *openInstrSession = VI_NULL;
    
    return currentStatus;
}

/*****************************************************************************/
/*=== END INSTRUMENT DRIVER SOURCE CODE =====================================*/
/*****************************************************************************/

